/***************************************************************************************************************//*!
*  
* @file INA.h
*
* @brief INA Class library header file
*
* @mainpage Arduino library to support the INAxxx family of current sensors
*
* @section Library_intro_section Description
*
* Class definition header for the INA class. This library gives a common interface to various INA power monitor
* devices, see https://github.com/SV-Zanshin/INA/wiki or the code below for a full list of currently supported
* devices.  The INA devices have a 3-5V power supply and, depending upon the model, can measure voltages up to
* 26V or 36V. They are devices with High-Side / Low-Side Measurement, Bi-Directional Current and Power Monitor
* with I2C Compatible Interface. The device documentation can be found at the following location:\n
* http://www.ti.com/amplifier-circuit/current-sense/power-current-monitors/products.html\n\n
* Detailed library descriptions are on the INA GitHub Wiki pages at https://github.com/SV-Zanshin/INA/wiki\n\n
* The INA devices, apart from the INA250 and INA260, require an external shunt of known resistance to be placed
* across the high-side or low-side supply or ground line and they use the small current generated by the shunt
* to compute the amperage passing across the circuit.  This value, coupled with the voltage measurement, allows
* the amperage and wattage to be computed by the INA device and these values can be read from the decives using
* the industry standard I2C protocol.
*
* @section license GNU General Public License v3.0
*
* This program is free software : you can redistribute it and/or modify it under the terms of the GNU General
* Public License as published by the Free Software Foundation, either version 3 of the License, or (at your
* option) any later version.This program is distributed in the hope that it will be useful, but WITHOUT ANY
* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.You should have received a copy of the GNU General Public License
* along with this program(see https://github.com/SV-Zanshin/INA/blob/master/LICENSE).  If not, see
* <http://www.gnu.org/licenses/>.
*
* @section author Author
*
* Written by Arnd\@SV-Zanshin
*
* @section versions Changelog
*
* Version | Date       | Developer                      | Comments
* ------- | ---------- | ------------------------------ | --------
* 1.0.8   | 2019-09-03 | https://github.com/miky2k      | Issue #43. Added new method "conversionFinisheds()"
* 1.0.8   | 2019-05-23 | https://github.com/avaldebe    | Issue #42. Restrict I2C scan to possible devices
* 1.0.8   | 2019-03-24 | https://github.com/mattlogic   | Issue #40. Corrected INA226_CONFIG_SADC_MASK value
* 1.0.8   | 2019-03-17 | https://github.com/Sv-Zanshin  | Issue #19. Corrected 4 value ranges in bus/shunt conversion
* 1.0.8   | 2019-02-16 | https://github.com/Sv-Zanshin  | Corrected and tested ESP32 implementation
* 1.0.8   | 2019-02-10 | https://github.com/Sv-Zanshin  | Issue #39. Allow non-AVR processors without EEPROM to run
* 1.0.8   | 2019-02-09 | https://github.com/Sv-Zanshin  | Cleaned up doxygen comment formatting in .h and .cpp files
* 1.0.8   | 2019-02-09 | https://github.com/Sv-Zanshin  | Issue #38. Add getDeviceAddress() function
* 1.0.7   | 2019-01-20 | https://github.com/Sv-Zanshin  | Issue #36&37. Changed for Travis-CI and automated doxygen
* 1.0.7   | 2018-12-27 | https://github.com/Sv-Zanshin  | Issue #33. Change program documentation to doxygen format
* 1.0.6   | 2018-12-13 | https://github.com/delboy711   | Issue #32. Incorrect ESP2866 rather than ESP8266
* 1.0.6   | 2018-10-19 | https://github.com/Sv-Zanshin  | Issue #31. Use full 0-32V Range on INA219 all the time
* 1.0.6   | 2018-10-19 | https://github.com/Sv-Zanshin  | Issue #30. Added TEENSY support & support large EEPROM
* 1.0.6   | 2018-10-14 | https://github.com/Sv-Zanshin  | Added correct wire handling for ESP8266 and ESP32
* 1.0.6   | 2018-10-07 | https://github.com/Sv-Zanshin  | Optimized getBusRaw() and getShuntRaw() functions
* 1.0.5   | 2018-10-04 | https://github.com/Sv-Zanshin  | Added getBusRaw() and getShuntRaw() functions
* 1.0.5   | 2018-09-29 | https://github.com/Sv-Zanshin  | Reformatted comments to different c++ coding style
* 1.0.4   | 2018-09-22 | https://github.com/Sv-Zanshin  | Issue #27. EEPROM Calls don't work with ESP32
* 1.0.4   | 2018-09-19 | https://github.com/Sv-Zanshin  | Issue #28. Overflow error when >31Amps specified in begin()
* 1.0.3   | 2018-09-04 | https://github.com/delboy711   | Issue #26. Incorrect INA3221 negative current readings
* 1.0.3   | 2018-08-18 | https://github.com/SV-Zanshin  | Issue #22. Reduce EEPROM Footprint
* 1.0.3   | 2018-08-18 | https://github.com/SV-Zanshin  | Issue #21. Rename I2C Constants to avoid redefine STM32F1
* 1.0.2   | 2018-07-22 | https://github.com/SV-Zanshin  | Issue #11. Reduce EEPROM footprint. Removed "deviceName", 
*                                                         8B. Changed datatype in structure to bit-level length 
*                                                         definitions, saving additional 3 bytes
* 1.0.2   | 2018-07-21 | https://github.com/avaldeve    | Issue #12. Incorrect const datatype for I2C Speeds
* 1.0.2   | 2018-07-12 | https://github.com/coelner     | Issue #9. Esplora doesn't accept "Wire.begin()"
* 1.0.2   | 2018-07-08 | https://github.com/SV-Zanshin  | Issue #2. Finished testing INA3221 across all functions
* 1.0.2   | 2018-07-07 | https://github.com/dnlwgnd     | Issue #4. Guard code used incorrect label
* 1.0.2   | 2018-06-30 | https://github.com/SV-Zanshin  | Issue #3. Adding faster I2C bus support
* 1.0.2   | 2018-06-29 | https://github.com/SV-Zanshin  | Issue #2. Adding INA3221 support to library
* 1.0.2   | 2018-06-29 | https://github.com/SV-Zanshin  | Issue #2. Adding INA3221 support to library
* 1.0.1   | 2018-06-24 | https://github.com/SV-Zanshin  | Removed extraneous elements from ina structure, optimzed code
* 1.0.1b  | 2018-06-23 | https://github.com/SV-Zanshin  | Fixed error on multiple devices with ina structure contents
* 1.0.1a  | 2018-06-23 | https://github.com/SV-Zanshin  | Removed debug mode and code
* 1.0.0   | 2018-06-22 | https://github.com/SV-Zanshin  | Initial release
* 1.0.0b  | 2018-06-17 | https://github.com/SV-Zanshin  | Continued coding, tested on INA219 and INA226
* 1.0.0a  | 2018-06-10 | https://github.com/SV-Zanshin  | Initial coding began
*
*******************************************************************************************************************/

/* Use old library if IDE is prior to V1.0 */
#if ARDUINO >= 100
  #include "Arduino.h"
#else
  #include "WProgram.h"
#endif

#ifndef INA__Class_h
  /*! Guard code definition to prevent multiple includes */
  #define INA__Class_h
  /*************************************************************************************************************//*!
  * Structure type definition contains a packed bit-level definition of information stored per device
  *****************************************************************************************************************/
  typedef struct 
  {
    uint8_t  type          :  4; ///< Values 0-15, see enumerated "ina_Type" for details
    uint8_t  operatingMode :  4; ///< Values 0-15, Default to continuous mode
    uint32_t address       :  7; ///< Values 0-127, I2C Address of device
    uint32_t maxBusAmps    :  7; ///< Values 0-127, Store initialization value
    uint32_t microOhmR     : 20; ///< Values 0-1.048.575, Store initialization value
  } inaEEPROM; // of structure

  /*************************************************************************************************************//*!
  * Structure type definition contains a packed bit-level definition of information stored on a device
  *****************************************************************************************************************/
  typedef struct inaDet : inaEEPROM 
  {
    uint8_t  busVoltageRegister   : 3; ///< 0- 7, Bus Voltage Register
    uint8_t  shuntVoltageRegister : 3; ///< 0- 7, Shunt Voltage Register
    uint8_t  currentRegister      : 3; ///< 0- 7, Current Register
    uint16_t shuntVoltage_LSB;         ///< Device dependent LSB factor
    uint16_t busVoltage_LSB;           ///< Device dependent LSB factor
    uint32_t current_LSB;              ///< Amperage LSB
    uint32_t power_LSB;                ///< Wattage LSB
    inaDet();                          ///< struct constructor
    inaDet(inaEEPROM inaEE);           ///< for ina = inaEE; assignment
  } inaDet; // of structure

  /*************************************************************************************************************//*!
  * Enumerated list detailing the names of all supported INA devices. The INA3221 is stored as 3 distinct devices
  * each with their own enumerated type.
  *****************************************************************************************************************/
  enum ina_Type { INA219, INA226, INA230, INA231, INA260, INA3221_0, INA3221_1, INA3221_2, INA_UNKNOWN };

  /*************************************************************************************************************//*!
  * Enumerated list detailing the operating modes of a given device
  *****************************************************************************************************************/
  enum ina_Mode 
  { 
    INA_MODE_SHUTDOWN,         ///< Device powered down
    INA_MODE_TRIGGERED_SHUNT,  ///< Triggered shunt, no bus
    INA_MODE_TRIGGERED_BUS,    ///< Triggered bus, no shunt
    INA_MODE_TRIGGERED_BOTH,   ///< Triggered bus and shunt
    INA_MODE_POWER_DOWN,       ///< shutdown or power-down
    INA_MODE_CONTINUOUS_SHUNT, ///< Continuous shunt, no bus
    INA_MODE_CONTINUOUS_BUS,   ///< Continuous bus, no shunt
    INA_MODE_CONTINUOUS_BOTH   ///< Both continuous, default value
  }; // of enumerated type

  /*****************************************************************************************************************
  ** Declare constants used in the class                                                                          **
  *****************************************************************************************************************/
  #ifndef I2C_MODES   // I2C related constants
    #define I2C_MODES // Guard code to prevent multiple definitions
    const uint32_t INA_I2C_STANDARD_MODE        =  100000; ///< Default normal I2C 100KHz speed
    const uint32_t INA_I2C_FAST_MODE            =  400000; ///< Fast mode
    const uint32_t INA_I2C_FAST_MODE_PLUS       = 1000000; ///< Really fast mode
    const uint32_t INA_I2C_HIGH_SPEED_MODE      = 3400000; ///< Turbo mode
  #endif
  const uint8_t  INA_CONFIGURATION_REGISTER     =       0; ///< Configuration Register address
  const uint8_t  INA_BUS_VOLTAGE_REGISTER       =       2; ///< Bus Voltage Register address
  const uint8_t  INA_POWER_REGISTER             =       3; ///< Power Register address
  const uint8_t  INA_CALIBRATION_REGISTER       =       5; ///< Calibration Register address
  const uint8_t  INA_MASK_ENABLE_REGISTER       =       6; ///< Mask enable Register address (not on all devices)
  const uint8_t  INA_ALERT_LIMIT_REGISTER       =       7; ///< Alert Limit Register address (not on all devices)
  const uint8_t  INA_MANUFACTURER_ID_REGISTER   =    0xFE; ///< Manuf. ID Register address (not on all devices)
  const uint8_t  INA_DIE_ID_REGISTER            =    0xFF; ///< Die ID Register address (not on all devices)
  const uint16_t INA_RESET_DEVICE               =  0x8000; ///< Write to configuration to reset device
  const uint16_t INA_CONVERSION_READY_MASK      =  0x0080; ///< Bit 4
  const uint16_t INA_CONFIG_MODE_MASK           =  0x0007; ///< Bits 0-3
  const uint16_t INA_ALERT_MASK                 =  0x03FF; ///< Mask off bits 0-9
  const uint8_t  INA_ALERT_SHUNT_OVER_VOLT_BIT  =      15; ///< Register bit
  const uint8_t  INA_ALERT_SHUNT_UNDER_VOLT_BIT =      14; ///< Register bit
  const uint8_t  INA_ALERT_BUS_OVER_VOLT_BIT    =      13; ///< Register bit
  const uint8_t  INA_ALERT_BUS_UNDER_VOLT_BIT   =      12; ///< Register bit
  const uint8_t  INA_ALERT_POWER_OVER_WATT_BIT  =      11; ///< Register bit
  const uint8_t  INA_ALERT_CONVERSION_RDY_BIT   =      10; ///< Register bit
  const uint8_t  INA_DEFAULT_OPERATING_MODE     =    B111; ///< Default continuous mode
  const uint8_t  INA219_SHUNT_VOLTAGE_REGISTER  =       1; ///< INA219 Shunt Voltage Register
  const uint8_t  INA219_CURRENT_REGISTER        =       4; ///< INA219 Current Register
  const uint16_t INA219_BUS_VOLTAGE_LSB         =     400; ///< INA219 LSB in uV *100 4.00mV
  const uint16_t INA219_SHUNT_VOLTAGE_LSB       =     100; ///< INA219 LSB in uV *10  10.0uV
  const uint16_t INA219_CONFIG_AVG_MASK         =  0x07F8; ///< INA219 Bits 3-6, 7-10
  const uint16_t INA219_CONFIG_PG_MASK          =  0xE7FF; ///< INA219 Bits 11-12 masked
  const uint16_t INA219_CONFIG_BADC_MASK        =  0x0780; ///< INA219 Bits 7-10  masked
  const uint16_t INA219_CONFIG_SADC_MASK        =  0x0038; ///< INA219 Bits 3-5
  const uint8_t  INA219_BRNG_BIT                =      13; ///< INA219 Bit for BRNG in config register
  const uint8_t  INA219_PG_FIRST_BIT            =      11; ///< INA219 first bit of Programmable Gain
  const uint8_t  INA226_SHUNT_VOLTAGE_REGISTER  =       1; ///< INA226 Shunt Voltage Register
  const uint8_t  INA226_CURRENT_REGISTER        =       4; ///< INA226 Current Register
  const uint16_t INA226_BUS_VOLTAGE_LSB         =     125; ///< INA226 LSB in uV *100 1.25mV
  const uint16_t INA226_SHUNT_VOLTAGE_LSB       =      25; ///< INA226 LSB in uV *10  2.5uV
  const uint16_t INA226_CONFIG_AVG_MASK         =  0x0E00; ///< INA226 Bits 9-11
  const uint16_t INA226_DIE_ID_VALUE            =  0x2260; ///< INA226 Hard-coded Die ID for INA226
  const uint16_t INA226_CONFIG_BADC_MASK        =  0x01C0; ///< INA226 Bits 6-8 masked
  const uint16_t INA226_CONFIG_SADC_MASK        =  0x0038; ///< INA226 Bits 3-4
  const uint8_t  INA260_SHUNT_VOLTAGE_REGISTER  =       0; ///< INA260 Register doesn't exist on device
  const uint8_t  INA260_CURRENT_REGISTER        =       1; ///< INA260 Current Register
  const uint16_t INA260_BUS_VOLTAGE_LSB         =     125; ///< INA260 LSB in uV *100 1.25mV
  const uint16_t INA260_CONFIG_BADC_MASK        =  0x01C0; ///< INA260 Bits 6-8  masked
  const uint16_t INA260_CONFIG_SADC_MASK        =  0x0038; ///< INA260 Bits 3-5  masked
  const uint8_t  INA3221_SHUNT_VOLTAGE_REGISTER =       1; ///< INA3221 Register number 1
  const uint16_t INA3221_BUS_VOLTAGE_LSB        =     800; ///< INA3221 LSB in uV *100 8mV
  const uint16_t INA3221_SHUNT_VOLTAGE_LSB      =     400; ///< INA3221 LSB in uV *10  40uV
  const uint16_t INA3221_CONFIG_BADC_MASK       =  0x01C0; ///< INA3221 Bits 7-10  masked
  const uint8_t  INA3221_MASK_REGISTER          =     0xF; ///< INA32219 Mask register
  const uint8_t  I2C_DELAY                      =      10; ///< Microsecond delay on I2C writes

  /*************************************************************************************************************//*!
  * @class   INA_Class
  * @brief   Forward definitions for the INA_Class
  *****************************************************************************************************************/
  class INA_Class 
  {
    public:
      INA_Class();
      ~INA_Class();
      uint8_t     begin                   (const uint8_t maxBusAmps, const uint32_t microOhmR, 
                                           const uint8_t deviceNumber = UINT8_MAX );
      void        setI2CSpeed             (const uint32_t i2cSpeed = INA_I2C_STANDARD_MODE);
      void        setMode                 (const uint8_t  mode,     const uint8_t deviceNumber = UINT8_MAX);
      void        setAveraging            (const uint16_t averages, const uint8_t deviceNumber = UINT8_MAX);
      void        setBusConversion        (const uint32_t convTime, const uint8_t deviceNumber = UINT8_MAX);
      void        setShuntConversion      (const uint32_t convTime, const uint8_t deviceNumber = UINT8_MAX);
      uint16_t    getBusMilliVolts        (const uint8_t deviceNumber = 0);
      uint16_t    getBusRaw               (const uint8_t deviceNumber = 0);
      int32_t     getShuntMicroVolts      (const uint8_t deviceNumber = 0);
      int16_t     getShuntRaw             (const uint8_t deviceNumber = 0);
      int32_t     getBusMicroAmps         (const uint8_t deviceNumber = 0);
      int32_t     getBusMicroWatts        (const uint8_t deviceNumber = 0);
      const char* getDeviceName           (const uint8_t deviceNumber = 0);
      uint8_t     getDeviceAddress        (const uint8_t deviceNumber = 0);
      void        reset                   (const uint8_t deviceNumber = 0);
      bool        conversionFinished      (const uint8_t deviceNumber = 0);
      void        waitForConversion       (const uint8_t deviceNumber = UINT8_MAX);
      bool        AlertOnConversion       (const bool alertState, const uint8_t deviceNumber = UINT8_MAX);
      bool        AlertOnShuntOverVoltage (const bool alertState, const int32_t milliVolts, const uint8_t deviceNumber = UINT8_MAX);
      bool        AlertOnShuntUnderVoltage(const bool alertState, const int32_t milliVolts, const uint8_t deviceNumber = UINT8_MAX);
      bool        AlertOnBusOverVoltage   (const bool alertState, const int32_t milliVolts, const uint8_t deviceNumber = UINT8_MAX);
      bool        AlertOnBusUnderVoltage  (const bool alertState, const int32_t milliVolts, const uint8_t deviceNumber = UINT8_MAX);
      bool        AlertOnPowerOverLimit   (const bool alertState, const int32_t milliAmps,  const uint8_t deviceNumber = UINT8_MAX);
    private:
      int16_t     readWord                (const uint8_t addr, const uint8_t deviceAddress);
      void        writeWord               (const uint8_t addr, const uint16_t data, const uint8_t deviceAddress);
      void        readInafromEEPROM       (const uint8_t deviceNumber);
      void        writeInatoEEPROM        (const uint8_t deviceNumber);
      void        initDevice              (const uint8_t deviceNumber);
      uint8_t     _DeviceCount =         0; ///< Total number of devices detected
      uint8_t     _currentINA  = UINT8_MAX; ///< Stores current INA device number
      inaEEPROM   inaEE;                    ///< INA device structure
      inaDet      ina;                      ///< INA device structure
      #if defined(__AVR__) || defined(CORE_TEENSY) || defined(ESP32) || defined(ESP8266) ||  (__STM32F1__)
      #else
        inaEEPROM _EEPROMEmulation[32];     ///< Actual array of up to 32 devices
      #endif
  }; // of INA_Class definition
#endif
